"""TreeBox widget"""
#************************************************************************************************************
# Author:            Arpad Kiss, arpadk@geometria.hu
# Modified:          1998.09.01.
# Platform:          -
# Description:       TreeBox version 0.74
#                    This widget uses load balance: loads items if it is needed. I use a 'levels' class 
#                    for this purpose. Look at the samples at the bottom of this module.
#                    This widget is static. You cannot change the behaviour of it. You can query the init 
#                    parameters, but you cannot set them.
#                    The space between items depends on the font size, so you can get overlapped items if the height of
#                    your icons or +/- signs is greater then the font size+linespace.
#                    If a branch has loaded then it won't be refreshed during the lifetime of the widget, except you
#                    call the Refresh method which deletes the subitems of the given item, and loads subitems just under
#                    the given item if this is in expanded state.
#                    You can Expand, Collapse and Select an item giving its key. Select is a little bit complicated,
#                    because you have to give the parents' keys if this item hasn't loaded yet.
#                    You can get the next parameters of the current item: Value, ItemKey, ItemData
#                    This widget may be used by mouse or keyboard.
#                    All variables/functions begining with an underscore are considered as local. Don't set them from
#                    outside of the class.
#************************************************************************************************************



#Imports
#************************************************************************************************************
from Tkinter import *
#************************************************************************************************************



#Constants
#************************************************************************************************************
_iexpanded_data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xf8, 0x1, 0xac, 0x2, 0x54, 0xfd, 0xac, 0xaa, 0x54, 0xd5, 0xff, 0xbf, 0x1, 0xe0, 0x1, 0xa0, 0x2, 0xc0, 0x2, 0xc0, 0x4, 0x80, 0xfc, 0xff};'
_iexpanded_maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xf8, 0x1, 0xfc, 0x3, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xf8, 0xff};'
_icollapsed_data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xf8, 0x1, 0x4, 0x2, 0x54, 0xfd, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0x4, 0x80, 0xfc, 0xff};'
_icollapsed_maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xf8, 0x1, 0xfc, 0x3, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0xfc, 0xff};'
_ileaf_data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xfc, 0x3f, 0x4, 0x20, 0xf4, 0x2f, 0x4, 0x20, 0xf4, 0x2f, 0x4, 0x20, 0xf4, 0x2f, 0x4, 0x20, 0xf4, 0x2f, 0x4, 0x20, 0xfc, 0x3f, 0x0, 0x0};'
_ileaf_maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0xfc, 0x3f, 0x0, 0x0};'
#************************************************************************************************************



#
#************************************************************************************************************
class TreeBox(Canvas):
    
    def __init__(self,
                 root=None,
		 levels,
		 rootfilter=None,
		 font='helvetica 10',
		 textcolor='black',
		 signsize=4,
		 signgap=24,
		 signcolor='gray40',
		 linecolor='gray50',
		 fontfunc=None,
		 colorfunc=None,
		 iconfunc=None,
		 visiblelines=YES,
		 visiblesigns=YES,
		 visibleicons=NO,
		 icongap=4,
		 linespace=2,
		 selcolor='yellow',
		 **kw):
	#
	Canvas.__init__(self,root,kw)
	self._topx,self._topy=5,15
	self._currentid=None
	self._items={}
	self._item_objects={}
	self._item_keys={}
	self._scrollbox_width=19
	#
	self._levels=levels
	self._rootfilter=rootfilter
	self._textfont=font
	self._textcolor=textcolor
	self._signsize=signsize
	self._signgap=signgap
	self._signcolor=signcolor
	self._linecolor=linecolor
	if fontfunc==None:
	    self._GetFont=lambda x,y=self._textfont: y
	else:
	    self._GetFont=fontfunc
	if colorfunc==None:
	    self._GetColor=lambda x,y=self._textcolor: y
	else:
	    self._GetColor=colorfunc
	self._icons={'expanded'  : BitmapImage(data=_iexpanded_data,maskdata=_iexpanded_maskdata,background='yellow',foreground='gray40'),
	             'collapsed' : BitmapImage(data=_icollapsed_data,maskdata=_icollapsed_maskdata,background='yellow',foreground='gray40'),
		     'leaf'      : BitmapImage(data=_ileaf_data,maskdata=_ileaf_maskdata,background='white',foreground='gray40')}
	if iconfunc==None:
	    self._GetIcon=_iconfunction
	else:
	    self._GetIcon=iconfunc
	self._visiblelines=visiblelines
	self._visiblesigns=visiblesigns
	self._visibleicons=visibleicons
	self._icongap=icongap
	self._tab=signsize*2+signgap
	self._linespace=linespace
	self._selected_color=selcolor
	#Scrollbars
	self._vbar=Scrollbar(self)
	self['yscrollcommand']=self._vbar.set
	self._vbar['command']=self.yview
	self._hbar=Scrollbar(self,orient=HORIZONTAL)
	self['xscrollcommand']=self._hbar.set
	self._hbar['command']=self.xview
	self._scrollbox_width=int(self._vbar.cget('width'))+int(self._vbar.cget('borderwidth'))*2
	#bottom_right corner
	self._brcorner=Frame(self,relief=RAISED,borderwidth=2)
	#bindings
	self.bind("<KeyPress-Return>",self._TreeBox_KeyPress_Return)
	self.bind("<KeyPress-Home>",self._TreeBox_KeyPress_Home)
	self.bind("<KeyPress-End>",self._TreeBox_KeyPress_End)
	self.bind("<KeyPress-Down>",self._TreeBox_KeyPress_Down)
	self.bind("<KeyPress-Up>",self._TreeBox_KeyPress_Up)
	self.bind("<KeyPress-Left>",self._TreeBox_KeyPress_Left)
	self.bind("<KeyPress-Right>",self._TreeBox_KeyPress_Right)
	self.bind("<KeyPress-Next>",self._TreeBox_KeyPress_PgDown)
	self.bind("<KeyPress-Prior>",self._TreeBox_KeyPress_PgUp)
	self.bind("<ButtonPress>",self._TreeBox_Click)
	self.bind("<Double-1>",self._TreeBox_DoubleClick)
	self.bind("<Configure>",self._Configure)
	#display level 0
	self._Expand(None)
	
    def __getitem__(self,i):
	return self.cget(i)
		       
    def configure(self,p={},**kw):
	self._PreConfig(p,kw)
	for k,v in p.items():
	    kw[k]=v
	Canvas.configure(self,kw)
	
    def cget(self,parName):
	try:
	    if parName=='Value':	    
		return self._items[self._currentid]['text']
	    elif parName=='ItemKey':
		return self._items[self._currentid]['key']
	    elif parName=='ItemData':
		return self._items[self._currentid]
	except:
	    return None
	if parName=='levels':
	    return self._levels
	elif parName=='rootfilter':
	    return self._rootfilter
	elif parName=='font':
	    return self._textfont
	elif parName=='textcolor':
	    return self._textcolor
	elif parName=='signsize':
	    return self._signsize
	elif parName=='signgap':
	    return self._signgap
	elif parName=='signcolor':
	    return self._signcolor
	elif parName=='linecolor':
	    return self._linecolor
	elif parName=='visiblelines':
	    return self._visiblelines
	elif parName=='visiblesigns':
	    return self._visiblesigns
	elif parName=='visibleicons':
	    return self._visibleicons
	elif parName=='icongap':
	    return self._icongap
	elif parName=='linespace':
	    return self._linespace
	elif parName=='selcolor':
	    return self._selected_color
	else:
	    # this is not mine, call the parent
	    return Canvas.cget(self,parName)
    
    #some properties have to been processed and  deleted, then the others are sent to Canvas
    # itt kell majd valahogy rpozicionlnom egy kvlrl megadott elemre
    def _PreConfig(self,p,kw):
	if kw.has_key('levels'):
	    del kw['levels']
	if p.has_key('levels'):
	    del p['levels']
	if kw.has_key('rootfilter'):
	    del kw['rootfilter']
	if p.has_key('rootfilter'):
	    del p['rootfilter']
	if kw.has_key('font'):
	    del kw['font']
	if p.has_key('font'):
	    del p['font']
	if kw.has_key('textcolor'):
	    del kw['textcolor']
	if p.has_key('textcolor'):
	    del p['textcolor']
	if kw.has_key('signsize'):
	    del kw['signsize']
	if p.has_key('signsize'):
	    del p['signsize']
	if kw.has_key('signgap'):
	    del kw['signgap']
	if p.has_key('signgap'):
	    del p['signgap']
	if kw.has_key('signcolor'):
	    del kw['signcolor']
	if p.has_key('signcolor'):
	    del p['signcolor']
	if kw.has_key('linecolor'):
	    del kw['linecolor']
	if p.has_key('linecolor'):
	    del p['linecolor']
	if kw.has_key('visiblelines'):
	    del kw['visiblelines']
	if p.has_key('visiblelines'):
	    del p['visiblelines']
	if kw.has_key('visiblesigns'):
	    del kw['visiblesigns']
	if p.has_key('visiblesigns'):
	    del p['visiblesigns']
	if kw.has_key('visibleicons'):
	    del kw['visibleicons']
	if p.has_key('visibleicons'):
	    del p['visibleicons']
	if kw.has_key('icongap'):
	    del kw['icongap']
	if p.has_key('icongap'):
	    del p['icongap']
	if kw.has_key('linespace'):
	    del kw['linespace']
	if p.has_key('linespace'):
	    del p['linespace']
	if kw.has_key('selcolor'):
	    del kw['selcolor']
	if p.has_key('selcolor'):
	    del p['selcolor']
    
    def _Configure(self,event):
	self._SetScrolling()

    def _TreeBox_Click(self,event):
	self.focus_set()
	ids=self.find_withtag(CURRENT)
	if len(ids)>0:
	    ItemID=ids[0]
	    tags=self.gettags(ItemID)
	    if 'itext' in tags:
		self._SetSelection(ItemID)
	    elif 'isign' in tags:
		if self._items[self._item_objects[ItemID]]['expanded']:
		    self._Collapse(self._item_objects[ItemID])
		else:
		    self._Expand(self._item_objects[ItemID])
	    elif 'iicon' in tags:
		self._SetSelection(self._item_objects[ItemID])
		
    def _TreeBox_KeyPress_Up(self,event):
	#up
	self._SetSelection(self._GetPrevious())		
		
    def _TreeBox_KeyPress_Down(self,event):
	#down
	self._SetSelection(self._GetNext())
		
    def _TreeBox_KeyPress_Left(self,event):
	#left
	if self._hbar.place_info():
	    self.xview('scroll',-1,'units')
		
    def _TreeBox_KeyPress_Right(self,event):
	#right
	if self._hbar.place_info():
	    self.xview('scroll',1,'units')
		
    def _TreeBox_KeyPress_PgUp(self,event):
	#pageup
	if self._vbar.place_info():
	    if self._hbar.place_info():
		ps=self._scrollbox_width+1
	    else:
		ps=0
	    y=self.bbox(self._currentid)[1]-float(self.winfo_height()-ps)*9/10
	    if y<0:
		ids=[]
		ids.append(self._firstitem)
	    else:
		ids=self.find_overlapping(0,y,self.bbox(ALL)[3]+self._scrollbox_width,y+self._linespace)
	    for iid in ids:
		if 'itext' in self.gettags(iid):
		    self._SetSelection(iid)
		    break
		
    def _TreeBox_KeyPress_PgDown(self,event):
	#pagedown
	if self._vbar.place_info():
	    if self._hbar.place_info():
		ps=self._scrollbox_width+1
	    else:
		ps=0
	    y=self.bbox(self._currentid)[1]+float(self.winfo_height()-ps)*9/10
	    if y>self.bbox(ALL)[3]-self._linespace:
		ids=[]
		ids.append(self._GetLast())
	    else:
		ids=self.find_overlapping(0,y,self.bbox(ALL)[3]+self._scrollbox_width,y+self._linespace)
	    for iid in ids:
		if 'itext' in self.gettags(iid):
		    self._SetSelection(iid)
		    break

    def _TreeBox_KeyPress_Home(self,event):
	#home
	self._SetSelection(self._firstitem)
		
    def _TreeBox_KeyPress_End(self,event):
	#end
	self._SetSelection(self._GetLast())

    def _TreeBox_KeyPress_Return(self,event):
	#enter
	if self._items[self._currentid]['expanded']:
	    self._Collapse(self._currentid)
	else:
	    self._Expand(self._currentid)

    def _TreeBox_DoubleClick(self,event):
	ids=self.find_withtag(CURRENT)
	if len(ids)>0:
	    tags=self.gettags(ids[0])
	    if 'itext' in tags:	    
		if self._items[ids[0]]['expanded']:
		    #it is expanded
		    self._Collapse()
		else:
		    #it is collapsed
		    self._Expand()
	    elif 'iicon' in tags:
		if self._items[self._item_objects[ids[0]]]['expanded']:
		    self._Collapse(self._item_objects[ids[0]])
		else:
		    self._Expand(self._item_objects[ids[0]])
		    
    def _SetSelection(self,ItemID):
	# deleting the old selection
	ids=self.find_withtag('selected')
	if len(ids)>0:
	    oldselID=ids[0]
	    self.delete(oldselID)
	    oldsitemID=self._item_objects[oldselID]
	    del self._item_objects[oldselID]
	    self._items[oldsitemID]['itemobject'].remove(oldselID)
	# setting the new
	x1,y1,x2,y2=self.bbox(ItemID)
	selID=self.create_rectangle(x1,y1,x2+5,y2,fill=self._selected_color,tags=(self._CreateLevelTag(ItemID),'selected','itemobject'))
	self._items[ItemID]['itemobject'].append(selID)
	self._item_objects[selID]=ItemID
	self.tkraise(ItemID,selID)
	self._currentid=ItemID
	#scroll to this item
	up,vy=self.yview()
	h=float(self.bbox(ALL)[3])+int(self._scrollbox_width)
	if self._hbar.place_info():
	    ps=self._scrollbox_width+1
	else:
	    ps=0
	if up>float(self.bbox(ItemID)[1]-self._linespace)/h:
	    self.yview('moveto',float(self.bbox(ItemID)[1])/h)
	elif self.winfo_height()<self.bbox(ItemID)[3]-h*up+2*int(self['borderwidth'])+3+ps:
	    self.yview('moveto',float(self.bbox(ItemID)[3]-self.winfo_height()+2*int(self['borderwidth'])+3+ps)/h)
	left,vx=self.xview()
	w=float(self.bbox(ALL)[2])+int(self._scrollbox_width)
	if left>float(self.bbox(ItemID)[0])/w:
	    self.xview('moveto',float(self.bbox(ItemID)[0])/(w+int(self._scrollbox_width)))
	elif vx<float(self.bbox(ItemID)[0]+int(self._scrollbox_width)-w*left)/(w-w*left):
	    self.xview('moveto',float(self.bbox(ItemID)[0])/(w+int(self._scrollbox_width)))
    
    # get the id of the item just above the current in the canvas
    def _GetPrevious(self,ItemID=None):
	if ItemID==None: ItemID=self._currentid
	if self._items[ItemID]['previousID']!=None:
	    iid=self._items[ItemID]['previousID']
	    while self._items[iid]['expanded']:
		iid=self._items[iid]['sitemslist'][len(self._items[iid]['sitemslist'])-1]
	    return iid
	elif self._items[ItemID]['parentID']!=None:
	    return self._items[ItemID]['parentID']
	else:
	    return ItemID
	
    # get the id of the item just under the current in the canvas
    def _GetNext(self,ItemID=None,firstcall=YES):
	if ItemID==None: ItemID=self._currentid
	if self._items[ItemID]['expanded'] and firstcall:
	    return self._items[ItemID]['sitemslist'][0]
	elif self._items[ItemID]['nextID']!=None:
	    return self._items[ItemID]['nextID']
	elif self._items[ItemID]['parentID']!=None and self._items[self._items[ItemID]['parentID']]['nextID']!=None:
	    return self._GetNext(self._items[ItemID]['parentID'],NO)
	else:
	    return ItemID

    # get the id of the last item in the canvas
    def _GetLast(self):
	iid=self._firstitem
	while self._items[iid]['nextID']!=None:
	    iid=self._items[iid]['nextID']
	while self._items[iid]['expanded']:
	    iid=self._items[iid]['sitemslist'][len(self._items[iid]['sitemslist'])-1]
	return iid
    
    # ParentKeys is needed when this item has not been loaded(left to right means up to down)
    def Select(self,ItemKey,ParentKeys=[]):
	if self._item_keys.has_key(ItemKey):
	    ids=self.find_all()
	    iid=self._item_keys[ItemKey]
	    if iid not in ids:
		iitem=self._items[iid]
		while iitem['parentID']!=None:
		    oldiid=iid
		    iid=iitem['parentID']
		    iitem=self._items[iid]
		    if iid in ids:
			if iitem['expanded']:
			    iid=oldiid
			break
		    iitem['expanded']=YES
		self._Expand(iid)
		iid=self._item_keys[ItemKey]
	else:
	    for ipkey in ParentKeys:
		if self._item_keys.has_key(ipkey):
		    iid=self._item_keys[ipkey]
		    if not self._items[iid]['expanded']:
			self._Expand(iid)
		else:
		    #bad parentkeys
		    return FALSE
	    if not self._item_keys.has_key(ItemKey): return FALSE
	iid=self._item_keys[ItemKey]
	self._SetSelection(iid)
	return TRUE
	    
    # ParentKeys is needed when this item has not been loaded(left to right means up to down)
    def Expand(self,ItemKey,ParentKeys=[]):
	if self.Select(ItemKey,ParentKeys):
	    iid=self._item_keys[ItemKey]
	    if not self._items[iid]['expanded']:
		self._Expand(iid)
	    return TRUE
	else:
	    return FALSE

    def Collapse(self,ItemKey):
	if self.Select(ItemKey):
	    iid=self._item_keys[ItemKey]
	    if self._items[iid]['expanded']:
		self._Collapse(iid)
	    return TRUE
	else:
	    return FALSE
	
    def _Expand(self,ItemID=CURRENT):
	tmpc=self['cursor']
	self['cursor']='watch'
	if ItemID==None:
	    #this is the implicit root, level 0
	    x,y=self._topx+self._tab,self._topy
	    items=self._levels.Items(0,self._rootfilter)
	    UpID=None
	    for iItem in items:
		newID=self.create_text(x,y,text=iItem[2],anchor=NW)
		x1,y1,x2,y2=self.bbox(newID)
		self.move(newID,0,y-y1)
		self._items[newID]={'text'        : iItem[2],
		                   'expanded'    : FALSE,
				   'sitemscount' : self._levels.Count(1,iItem[1]),
				   'sitemslist'  : [],
				   'level'       : 0,
				   'key'         : iItem[1],
				   'extradatas'  : iItem[3],
				   'previousID'  : UpID,
				   'nextID'      : None,
				   'parentID'    : None}
		self._CreateItemObject(newID,newID,UpID)
		self._item_keys[iItem[1]]=newID
		if UpID!=None:
		    self._items[UpID]['nextID']=newID
		else:
		    self._SetSelection(newID)
		    self._firstitem=newID
		UpID=newID
		y=self.bbox(newID)[3]+self._linespace
	    self._SetScrolling()

	else:
	    # level 1,2,....
	    # to get the real item Id(instead of CURRENT)
	    ItemID=self.find_withtag(ItemID)[0]
	    ItemKey=self._items[ItemID]['key']
	    if self._items[ItemID]['sitemscount']!=0:
		# it has subitems
		# we have to make room for the new items
		# at first we add 'move' tag to items under ItemID
		self._SetMoveTag(ItemID)
		# creating the new text items	
		if len(self._items[ItemID]['sitemslist'])==0:
		    #list of subitem keys hasn't been filled up
		    items=self._levels.Items(self._items[ItemID]['level']+1,ItemKey)
		    x,y=self.bbox(ItemID)[0]+self._tab,self.bbox(ItemID)[3]+self._linespace
		    UpID=ItemID
		    PrevID=None
		    self._items[ItemID]['sitemscount']=len(items)
		    space=0
		    for iItem in items:
			newID=self.create_text(x,y,text=iItem[2],anchor=NW)
			x1,y1,x2,y2=self.bbox(newID)
			self.move(newID,0,y-y1)
			self._items[newID]={'text'        : iItem[2],
			                   'expanded'    : FALSE,
					   'sitemscount' : self._levels.Count(self._items[ItemID]['level']+2,iItem[1]),
					   'sitemslist'  : [],
					   'level'       : self._items[ItemID]['level']+1,
					   'key'         : iItem[1],
					   'extradatas'    : iItem[3],
					   'previousID'  : PrevID,
					   'nextID'      : None,
					   'parentID'    : ItemID}
			self._items[ItemID]['sitemslist'].append(newID)
			self._CreateItemObject(newID,newID,UpID)
			self._item_keys[iItem[1]]=newID
			space=space+abs(self.bbox(newID)[3]-self.bbox(newID)[1])+self._linespace
			UpID=newID
			if PrevID!=None:
			    self._items[PrevID]['nextID']=newID
			PrevID=newID
			y=self.bbox(UpID)[3]+self._linespace
		else:
		    #list of subitem keys has been filled up
		    items=self._items[ItemID]['sitemslist']
		    x,y=self.bbox(ItemID)[0]+self._tab,self.bbox(ItemID)[3]+self._linespace
		    tmp_sitemslist=[]
		    UpID=ItemID
		    PrevID=None
		    space=0
		    for iItem in items:
			newID=self.create_text(x,y,text=self._items[iItem]['text'],anchor=NW)
			x1,y1,x2,y2=self.bbox(newID)
			self.move(newID,0,y-y1)
			self._items[iItem]['parentID']=ItemID
			self._items[iItem]['previousID']=PrevID
			self._items[iItem]['nextID']=None
			self._CreateItemObject(newID,iItem,UpID)
			UpID=newID
			if PrevID!=None:
			    self._items[PrevID]['nextID']=newID
			PrevID=newID
			y=self.bbox(UpID)[3]+self._linespace
			self._items[newID]=self._items[iItem]
			if self._currentid==iItem:
			    self._SetSelection(newID)
			tmp_sitemslist.append(newID)
			self._item_keys[self._items[newID]['key']]=newID
			space=space+abs(self.bbox(newID)[3]-self.bbox(newID)[1])+self._linespace
			del self._items[iItem]
		    self._items[ItemID]['sitemslist']=tmp_sitemslist
		# we have to make room for the new items
		# we are moving items with 'move' tag
		self._MoveItemObjects(ItemID,space)
		# if these were expanded, then we have to expand them    
		for iItemID in self._items[ItemID]['sitemslist']:
		    if self._items[iItemID]['expanded']:
			#it is expanded, so we have to call this function again
			self._Expand(iItemID)
		# finally we set the expanded flag
		self._SetExpanded(ItemID,TRUE)
	self['cursor']=tmpc
		
    #LeaveExpanded=TRUE when we call this function to hide the subbranches
    def _Collapse(self,ItemID=CURRENT,LeaveExpanded=FALSE):
	ItemID=self.find_withtag(ItemID)[0]
	tmpc=self['cursor']
	self['cursor']='watch'
	# deleting the subitems, if this has
	# looping throught the subitems level
	items=self._items[ItemID]['sitemslist']
	# we add 'move' to items under ItemID
	self._SetMoveTag(ItemID)
	space=0
	for iItemID in items:
	    space=space+abs(self.bbox(iItemID)[3]-self.bbox(iItemID)[1])+self._linespace
	    if  self._items[iItemID]['expanded']:
		#it is expanded, so we have to call this function again
		self._Collapse(iItemID,TRUE)
	    self._DeleteItemObject(iItemID)
	# we have to pull up the items with 'move' tag
	self._MoveItemObjects(ItemID,-space)
	if not LeaveExpanded:
	    # set the expanded flag to FALSE
	    self._SetExpanded(ItemID,FALSE)
	self['cursor']=tmpc
	    
    def _CreateLevelTag(self,ItemID=None):
	#'L*Level 0*parent key*'
	if self._items[ItemID]['parentID']==None:
	    return 'L*0**'
	else:
	    return 'L*'+str(self._items[ItemID]['level'])+'*'+self._items[self._items[ItemID]['parentID']]['key']+'*' 	

    def _CreateSubItemsLevelTag(self,ParentItemID):
	    return 'L*'+str(self._items[ParentItemID]['level']+1)+'*'+self._items[ParentItemID]['key']+'*' 	

    def _SetMoveTag(self,ItemID):
	items=self.find_withtag('itemobject')
	iy2=self.bbox(ItemID)[3]	
	# moving
	for i in items:
	    y2=self.bbox(i)[3]	
	    if y2>iy2 and self._item_objects[i]!=ItemID:
		self.addtag_withtag('move'+str(ItemID),i)
	# redrawing vertical lines    
	tmpID=ItemID    
	while 1:
	    for i in self._items[tmpID]['vlines']:
		x1,y1,x2,y2=self.coords(i)
		if y1>y2: y1,y2=y2,y1
		if y2>iy2:
		    self.addtag_withtag('redraw'+str(ItemID),i)
	    if self._items[tmpID]['parentID']==None: break
	    tmpID=self._items[tmpID]['parentID']
	
    def _MoveItemObjects(self,ItemID,space):
	items=self.find_withtag('move'+str(ItemID))
	# moving
	for i in items:
	    self.move(i,0,space)
	# redrawing vertical lines    
	items=self.find_withtag('redraw'+str(ItemID))
	for i in items:
	    x1,y1,x2,y2=self.coords(i)
	    if y1>y2: y1,y2=y2,y1
	    sc=(y2-y1+space)/(y2-y1)
	    self.scale(i,x2,y2,1,sc)
	# delete the 'move' and 'redraw' tag
	self.dtag('move'+str(ItemID),'move'+str(ItemID))
	self.dtag('redraw'+str(ItemID),'redraw'+str(ItemID))
	# set the scrolled region
	self._SetScrolling()
	
    # the ItemID and iIndex are different when the self._items is filled, but the parent is collapsed and we want to
    # expand it
    def _CreateItemObject(self,ItemID,iItems,UpID):
	self._items[iItems]['itemobject']=[ItemID]
	self._items[iItems]['vlines']=[]
	self.itemconfigure(ItemID,tags=(self._CreateLevelTag(iItems),'itext','itemobject'),font=self._GetFont(self._items[iItems],self._textfont),fill=self._GetColor(self._items[iItems],self._textcolor))
	self._item_objects[ItemID]=ItemID
	self._DrawLines(ItemID,iItems,UpID)
	self._UpdateSign(ItemID,iItems)
	self._UpdateIcon(ItemID,iItems)
	
    def _DeleteItemObject(self,ItemID):
	self.delete(ItemID)
	for i in self._items[ItemID]['itemobject']:
	    self.delete(i)
	    del self._item_objects[i]
	    if self._items[ItemID]['parentID']!=None and i in self._items[self._items[ItemID]['parentID']]['vlines']:
		self._items[self._items[ItemID]['parentID']]['vlines'].remove(i)
	self._items[ItemID]['itemobject']=[]
	self._items[ItemID]['vlines']=[]

    def _SetExpanded(self,ItemID,Value):
	self._items[ItemID]['expanded']=Value
	self._UpdateSign(ItemID,ItemID)
	self._UpdateIcon(ItemID,ItemID)

    def _UpdateSign(self,ItemID,iItems):
	if not self._visiblesigns: return
	#deleting the old sign
	items=self.find_withtag('isign')
	for iItemID in items:
	    if iItems==self._item_objects[iItemID]:
		self.delete(iItemID)
		del self._item_objects[iItemID]
		self._items[iItems]['itemobject'].remove(iItemID)
	#creating new one
	x,y=self.bbox(ItemID)[0]-self._signgap-self._signsize,self.bbox(ItemID)[1]+(self.bbox(ItemID)[3]-self.bbox(ItemID)[1])/2
	if self._items[iItems]['sitemscount']!=0:
	    # box
	    newID=self.create_polygon(x-self._signsize,y-self._signsize,x+self._signsize,y-self._signsize,x+self._signsize,y+self._signsize,x-self._signsize,y+self._signsize,tags=(self._CreateLevelTag(iItems),'isign','itemobject'),outline=self._linecolor,fill='')
	    self._items[iItems]['itemobject'].append(newID)
	    self._item_objects[newID]=ItemID
	    if self._items[iItems]['expanded']:
		# - sign
		newID=self.create_line(x-self._signsize+2,y,x+self._signsize-1,y,tags=(self._CreateLevelTag(iItems),'isign','itemobject'),fill=self._signcolor)
		self._items[iItems]['itemobject'].append(newID)
		self._item_objects[newID]=ItemID
	    else:
		# + sign
		newID=self.create_line(x-self._signsize+2,y,x+self._signsize-1,y,tags=(self._CreateLevelTag(iItems),'isign','itemobject'),fill=self._signcolor)
		self._items[iItems]['itemobject'].append(newID)
		self._item_objects[newID]=ItemID
		newID=self.create_line(x,y-self._signsize+2,x,y+self._signsize-1,tags=(self._CreateLevelTag(iItems),'isign','itemobject'),fill=self._signcolor)
		self._items[iItems]['itemobject'].append(newID)
		self._item_objects[newID]=ItemID

    def _DrawLines(self,ItemID,iItems,UpID):
	if not self._visiblelines: return
	#creating the horizontal one
	x,y=self.bbox(ItemID)[0],self.bbox(ItemID)[1]+(self.bbox(ItemID)[3]-self.bbox(ItemID)[1])/2
	if self._items[iItems]['sitemscount']!=0:
	    newID=self.create_line(x-self._signgap,y,x,y,tags=(self._CreateLevelTag(iItems),'line','ihline','itemobject'),fill=self._linecolor)
	else:
	    newID=self.create_line(x-self._signgap-self._signsize,y,x,y,tags=(self._CreateLevelTag(iItems),'line','ihline','itemobject'),fill=self._linecolor)
	self._items[iItems]['itemobject'].append(newID)
	self._item_objects[newID]=ItemID
	#creating the vertical one
	if UpID!=None:
	    # there is a item above this
	    if self._items[UpID]['level']==self._items[iItems]['level']:		
		# they are on the same level
		if self._items[UpID]['sitemscount']>0:
		    if self._items[iItems]['sitemscount']!=0:
		        newID=self.create_line(x-self._signgap-self._signsize,y-self._signsize,x-self._signgap-self._signsize,self.bbox(UpID)[1]+(self.bbox(UpID)[3]-self.bbox(UpID)[1])/2+self._signsize,tags=(self._CreateLevelTag(iItems),'line','ivline','itemobject'),fill=self._linecolor)
		    else:
			newID=self.create_line(x-self._signgap-self._signsize,y,x-self._signgap-self._signsize,self.bbox(UpID)[1]+(self.bbox(UpID)[3]-self.bbox(UpID)[1])/2+self._signsize,tags=(self._CreateLevelTag(iItems),'line','ivline','itemobject'),fill=self._linecolor)
		else:
		    if self._items[iItems]['sitemscount']!=0:
			newID=self.create_line(x-self._signgap-self._signsize,y-self._signsize,x-self._signgap-self._signsize,self.bbox(UpID)[1]+(self.bbox(UpID)[3]-self.bbox(UpID)[1])/2,tags=(self._CreateLevelTag(iItems),'line','ivline','itemobject'),fill=self._linecolor)
		    else:
			newID=self.create_line(x-self._signgap-self._signsize,y,x-self._signgap-self._signsize,self.bbox(UpID)[1]+(self.bbox(UpID)[3]-self.bbox(UpID)[1])/2,tags=(self._CreateLevelTag(iItems),'line','ivline','itemobject'),fill=self._linecolor)
	    else:
		if self._items[iItems]['sitemscount']!=0:
		    newID=self.create_line(x-self._signgap-self._signsize,y-self._signsize,x-self._signgap-self._signsize,self.bbox(UpID)[3],tags=(self._CreateLevelTag(iItems),'line','ivline','itemobject'),fill=self._linecolor)
		else:
		    newID=self.create_line(x-self._signgap-self._signsize,y,x-self._signgap-self._signsize,self.bbox(UpID)[3],tags=(self._CreateLevelTag(iItems),'line','ivline','itemobject'),fill=self._linecolor)
	    #
	    
	    self._items[iItems]['itemobject'].append(newID)
	    self._item_objects[newID]=ItemID
	    self._items[UpID]['vlines'].append(newID)

    def _UpdateIcon(self,ItemID,iItems):
	if not self._visibleicons: return
	#deleting the old icon
	items=self.find_withtag('iicon')
	for iItemID in items:
	    if iItems==self._item_objects[iItemID]:
		self.delete(iItemID)
		del self._item_objects[iItemID]
		self._items[iItems]['itemobject'].remove(iItemID)
	#creating new one
	x,y=self.bbox(ItemID)[0]-self._icongap,self.bbox(ItemID)[1]+(self.bbox(ItemID)[3]-self.bbox(ItemID)[1])/2
	newID=self.create_image(x,y,image=self._GetIcon(self._items[iItems],self._icons),tags=(self._CreateLevelTag(iItems),'iicon','itemobject'),anchor=E)
	self._items[iItems]['itemobject'].append(newID)
	self._item_objects[newID]=ItemID
	
    def _SetScrolling(self):
	self['scrollregion']='0 0 '+str(self.bbox(ALL)[2]+int(self._scrollbox_width))+' '+str(self.bbox(ALL)[3]+int(self._scrollbox_width))
	if (self.yview()[1]<1 or self.yview()[0]!=0) and (self.xview()[1]<1 or self.xview()[0]!=0):
	    self._vbar.place(relx=1,x=-int(self['borderwidth']),rely=0,y=int(self['borderwidth']),relheight=1,height=-2*int(self['borderwidth'])-self._scrollbox_width,anchor=NE)
	    self._vbar.set(self.yview()[0],self.yview()[1])
	    self._hbar.place(relx=0,x=int(self['borderwidth']),rely=1,y=-int(self['borderwidth']),relwidth=1,width=-2*int(self['borderwidth'])-self._scrollbox_width,anchor=SW)
	    self._hbar.set(self.xview()[0],self.xview()[1])
	    self._brcorner.place(relx=1,x=-int(self['borderwidth']),rely=1,y=-int(self['borderwidth']),width=self._scrollbox_width,height=self._scrollbox_width,anchor=SE)
	elif (self.yview()[1]<1 or self.yview()[0]!=0) and self.xview()[1]==1 and self.xview()[0]==0:
	    self._vbar.place(relx=1,x=-int(self['borderwidth']),rely=0,y=int(self['borderwidth']),relheight=1,height=-2*int(self['borderwidth']),anchor=NE)
	    self._vbar.set(self.yview()[0],self.yview()[1])
	    self._hbar.place_forget()
	    self._brcorner.place_forget()
	elif self.yview()[1]==1 and self.yview()[0]==0 and (self.xview()[1]<1 or self.xview()[0]!=0):
	    self._vbar.place_forget()
	    self._hbar.place(relx=0,x=int(self['borderwidth']),rely=1,y=-int(self['borderwidth']),relwidth=1,width=-2*int(self['borderwidth']),anchor=SW)
	    self._hbar.set(self.xview()[0],self.xview()[1])
	    self._brcorner.place_forget()
	else:
	    self._vbar.place_forget()
	    self._hbar.place_forget()
	    self._brcorner.place_forget()

    def Refresh(self,ItemKey):
	tmpc=self['cursor']
	self['cursor']='watch'
	ItemID=self._item_keys[ItemKey]
	itwasexpanded=NO
	if self._items[ItemID]['expanded']:
	    self._Collapse(ItemID)
	    itwasexpanded=YES
	self._DelSubItems(ItemID)
	self._items[ItemID]['sitemscount']=  self._levels.Count(self._items[ItemID]['level']+1,self._items[ItemID]['key'])
	if itwasexpanded:
	    self._Expand(ItemID)
	self['cursor']=tmpc

    def _DelSubItems(self,ItemID):
	for iid in self._items[ItemID]['sitemslist']:
	    self._DelSubItems(iid)
	    del self._items[iid]
	self._items[ItemID]['sitemslist']=[]
#************************************************************************************************************



# default icon function
#************************************************************************************************************
def _iconfunction(itemdict,icons):
    if itemdict['expanded']:
	#get expanded icon
	return icons['expanded']
    elif itemdict['sitemscount']!=0:
	#get collapsed icon
	return icons['collapsed']
    else:
	#get leaf icon
	return icons['leaf']
#************************************************************************************************************



# base levels class
#************************************************************************************************************
class levels:

    def __init__(self):
	pass
    
    def Count(self,level,key=None):
	return 0
    
    def Items(self,level,key=None):
	return ()  #((parent,key,text,extradatas),....)
#************************************************************************************************************



#Test
#************************************************************************************************************
if __name__=='__main__':
    # static tuple levels class
    class tlevels(levels):

	def __init__(self,ts):
	    self.ts=ts
	
	def Count(self,level,key=None):
	    if level>len(self.ts)-1: return 0
	    if key==None:
		return len(self.ts[level])
	    else:
		return len(filter(lambda parelem,parkey=key: parelem[0]==parkey ,self.ts[level]))
	
	def Items(self,level,key=None):
	    if level>len(self.ts)-1: return 0
	    if key==None:
		return self.ts[level]
	    else:
		return filter(lambda parelem,parkey=key: parelem[0]==parkey ,self.ts[level])
    
    #file system levels class     
    import os
    class fslevels(levels):
	
	def Count(self,level,key=None):
	    try:
		if key==None: 
		    return len(os.listdir(os.sep))
		else:
		    return len(os.listdir(key))
	    except:
		return 0
	
	def Items(self,level,key=None):
	    try:
		if key==None: 
		    return map(lambda parelem: (os.sep,os.sep+parelem,parelem,None), os.listdir(os.sep))
		else:
		    return map(lambda parelem,parkey=key: (parkey,parkey+os.sep+parelem,parelem,None), os.listdir(key))
	    except:
		return ()
    
    # font&color sample functions
    def fontfunction(itemdict,deffont):
	if itemdict['level']==0:
	    # the first level uses the default font
	    return deffont
	else:
	    if 22-4*itemdict['level']<10:
		return 'courier 10 bold italic'
	    else:
		return 'courier '+str(22-4*itemdict['level'])+' bold italic'
    
    def colorfunction(itemdict,defcolor):
	if itemdict['level']==0: 
	    # the first level uses the default color
	    return defcolor
	else:
	    if 2*itemdict['level']*10>60:
		return 'gray60'
	    else:
		return 'gray'+ str(2*itemdict['level']*10)

    root=Tk()
    
    #a simple treebox
    l0=((None,'l11','Trees',None),(None,'l12','Animals',None),(None,'l13','Things',None))  #((parent,key,text,extra datas),....)
    l1=(('l11','l21','Oak',None),('l11','l22','Chestnut',None),('l13','l23','Cars',None),('l13','l24','Pencil',None),('l13','l25','Hammer',None),('l13','l26','Table',None))
    l2=(('l23','l31','Ford',None),('l23','l32','Chevrolet',None))
    def iconfunction(itemdict,icons):
	if itemdict['text']=='Trees':
	    try:
		if itemdict['expanded']:		
		    return icons['btree']
		else:
		    return icons['tree']
	    except:
		if itemdict['expanded']:		
		    data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0x88, 0x8, 0x32, 0x15, 0x44, 0x55, 0xea, 0x32, 0x90, 0xa, 0xf8, 0x3e, 0xc6, 0x47, 0xb1, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
		    maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xd8, 0xa, 0xfe, 0x37, 0x76, 0x7d, 0xfe, 0xbf, 0xbf, 0x6b, 0xff, 0xfe, 0xff, 0x7f, 0xbe, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
		    icons['treeinb']= BitmapImage(data=data,maskdata=maskdata,background='pink',foreground='brown')
		    return icons['treeinb']
		else:
		    data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0x88, 0x8, 0x32, 0x15, 0x44, 0x55, 0xea, 0x32, 0x90, 0xa, 0xf8, 0x3e, 0xc6, 0x47, 0xb1, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
		    maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xd8, 0xa, 0xfe, 0x37, 0x76, 0x7d, 0xfe, 0xbf, 0xbf, 0x6b, 0xff, 0xfe, 0xff, 0x7f, 0xbe, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
		    icons['tree']= BitmapImage(data=data,maskdata=maskdata,background='green',foreground='brown')
		    return icons['tree']
	elif itemdict['key'] in ['l21','l22']:
	    try:
		return icons['tree']
	    except:
		data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0x88, 0x8, 0x32, 0x15, 0x44, 0x55, 0xea, 0x32, 0x90, 0xa, 0xf8, 0x3e, 0xc6, 0x47, 0xb1, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
		maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0xd8, 0xa, 0xfe, 0x37, 0x76, 0x7d, 0xfe, 0xbf, 0xbf, 0x6b, 0xff, 0xfe, 0xff, 0x7f, 0xbe, 0x1f, 0xc0, 0x3, 0xe0, 0x3, 0x60, 0x7, 0x50, 0xb};'
		icons['tree']= BitmapImage(data=data,maskdata=maskdata,background='green',foreground='brown')
		return icons['tree']
	elif itemdict['key'] in ['l31','l32']:
	    try:
		return icons['carleaf']
	    except:
		icons['carleaf']= BitmapImage(data=_ileaf_data,maskdata=_ileaf_maskdata,background='red',foreground='black')
		return icons['carleaf']
	elif itemdict['text']=='Cars':
	    try:
		return icons['car']
	    except:
		data= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0x80, 0x0, 0x20, 0x0, 0x20, 0x0, 0x90, 0xf, 0x48, 0x12, 0x28, 0x22, 0xfc, 0x7f, 0x12, 0x42, 0x19, 0x99, 0xef, 0xf7, 0x34, 0x2c, 0x18, 0x18};'
		maskdata= '#define bmp_width 16'+chr(10)+'#define bmp_height 12'+chr(10)+'static char bmp_bits[] = {'+chr(10)+'0x80, 0x0, 0x20, 0x0, 0x20, 0x0, 0x90, 0xf, 0x48, 0x12, 0x28, 0x22, 0xfc, 0x7f, 0xfe, 0x7f, 0xff, 0xff, 0xff, 0xff, 0x3c, 0x3c, 0x18, 0x18};'
		icons['car']= BitmapImage(data=data,maskdata=maskdata,background='red',foreground='black')
		return icons['car']
	elif itemdict['expanded']:
	    #get expanded icon
	    return icons['expanded']
	elif itemdict['sitemscount']!=0:
	    #get collapsed icon
	    return icons['collapsed']
	else:
	    #get leaf icon
	    return icons['leaf']
    trb=TreeBox(root,tlevels((l0,l1,l2)),visibleicons=YES,iconfunc=iconfunction)
    trb.place(relx=0,rely=0,relwidth=0.33,relheight=0.9)
    trb.Select('l32',['l13','l23'])
    def Select():
	if trb.cget('ItemData')!=None:
	    trb.Select(trb.cget('ItemData')['key'])
    def Expand():
	if trb.cget('ItemData')!=None:
	    trb.Expand(trb.cget('ItemData')['key'])
    def Collapse():
	if trb.cget('ItemData')!=None:
	    trb.Collapse(trb.cget('ItemData')['key'])
    btn=Button(root,text='Show selected',command=Select)
    btn.place(x=0,rely=1,relwidth=0.111,relheight=0.1,anchor=SW)
    btn=Button(root,text='Expand',command=Expand)
    btn.place(relx=0.111,rely=1,relwidth=0.111,relheight=0.1,anchor=SW)
    btn=Button(root,text='Collapse',command=Collapse)
    btn.place(relx=0.222,rely=1,relwidth=0.111,relheight=0.1,anchor=SW)

    #file system browser&refresh button
    fsb=TreeBox(root,fslevels(),font='courier 22 italic bold',textcolor='purple',linecolor='indianred',signcolor='darkblue',signsize=6,fontfunc=fontfunction,colorfunc=colorfunction,relief=GROOVE,borderwidth=5,selcolor='gray80',linespace=10)
    fsb.configure(bg='lightpink')
    fsb.place(relx=0.33,rely=0,relwidth=0.33,relheight=0.9)
    def Refresh():
	if fsb.cget('ItemData')!=None:
	    fsb.Refresh(fsb.cget('ItemData')['key'])
    btn=Button(root,text='Refresh',command=Refresh)
    btn.place(relx=0.333,rely=1,relwidth=0.33,relheight=0.1,anchor=SW)

    #another file system browser
    fsb2=TreeBox(root,fslevels(),font='courier 22 italic bold',textcolor='purple',linecolor='indianred',signcolor='darkblue',signsize=6,fontfunc=fontfunction,colorfunc=colorfunction,visiblelines=FALSE,relief=RAISED,borderwidth=5,selcolor='lightgreen',visibleicons=YES)
    fsb2.configure(bg='lightpink')
    fsb2.place(relx=0.66,rely=0,relwidth=0.33,relheight=1)

    root.geometry('640x480')
    root.mainloop()    

    print 'Value:',trb['Value']
    print 'Key:',trb.cget('ItemKey')
    print 'ItemData:',trb['ItemData']
    print '*'*20
    print 'Value:',fsb.cget('Value')
    print 'Key:',fsb.cget('ItemKey')
    print 'ItemData:',fsb['ItemData']
    print '*'*20
    print 'Value:',fsb2['Value']
    print 'Key:',fsb2.cget('ItemKey')
    print 'ItemData:',fsb2.cget('ItemData')
